/******************************************************/
/* Examples Program For "ET-BASE dsPIC30F2010"  Board */
/* Hardware    : ET-BASE dsPIC30F2010                 */
/* Target MCU  : dsPIC30F2010                         */
/*       	   : X-TAL : 7.3728 MHz                   */
/*             : Run 117.9648MHz                      */
/*             : Selec OSC Mode = XT w/PLL 16x        */
/* Compiler    : MPLAB v8.36 + C30 v3.12       	      */
/* Last Update : 12/October/2009                      */
/******************************************************/
/* Function    : QC Test Hardware Board System        */
/******************************************************/
/* UART1                                              */
/* -> Used RC13 = TXD1                                */
/* -> Used RC14 = RXD1                                */
/******************************************************/
/* LCD 16x2                                           */
/* -> D4 = RB0					  				      */ 
/* -> D5 = RB1					   				      */
/* -> D6 = RB2					   				      */
/* -> D7 = RB3					   				      */
/* -> RS = RB4					   				      */
/* -> EN = RB5				     				      */
/******************************************************/
/* GPIO                                               */
/* -> RB[0..5]                                        */
/* -> RE[0..5],RE8                                    */
/* -> RF2,RF3                                         */
/* -> RC13,RC14,RD0,RD1                               */
/******************************************************/

#include <p30f2010.h>                                		// dsPIC30F2010 MPU Register
#include <stdio.h>											// Used "sprintf" Function
#include "uart.h"											// Used UART Config

/* Setup Configuration For ET-BASE dsPIC30F2010 */
_FOSC(CSW_FSCM_OFF & XT_PLL16);								// Disable Clock Switching,Enable Fail-Salf Clock
                                                            // Closk Source = Primary XT + (PLL x 16)
_FWDT(WDT_OFF);												// Disable Watchdog 
_FBORPOR(PBOR_ON & BORV_45 & PWRT_64 & MCLR_EN);			// Enable Brown-Out = 4.5V,Power ON = 64mS,Enable MCLR
_FGS(CODE_PROT_OFF);										// Code Protect OFF
/* End   Configuration For ET-BASE dsPIC30F2010 */

char uart_buf1[40]; 										// "sprintf" Buffer

// Character LCD Interface Pins 
#define TRIS_DATA_PIN_4     TRISBbits.TRISB0				// Direction D4
#define TRIS_DATA_PIN_5     TRISBbits.TRISB1				// Direction D5
#define TRIS_DATA_PIN_6     TRISBbits.TRISB2				// Direction D6
#define TRIS_DATA_PIN_7     TRISBbits.TRISB3				// Direction D7
#define TRIS_RS             TRISBbits.TRISB4   				// Direction RS
#define TRIS_E              TRISBbits.TRISB5 				// Direction E

#define DATA_PIN_4          LATBbits.LATB0					// RB0 = D4 LCD
#define DATA_PIN_5          LATBbits.LATB1					// RB1 = D5 LCD
#define DATA_PIN_6          LATBbits.LATB2					// RB2 = D6 LCD
#define DATA_PIN_7          LATBbits.LATB3					// RB3 = D7 LCD
#define RS_PIN              LATBbits.LATB4     				// RB4 = RS LCD
#define E_PIN               LATBbits.LATB5     				// RB5 = E  LCD
								
/* Display ON/OFF Control */
#define DON                 0x0F  							// Display on      
#define DOFF                0x0B  							// Display off     
#define CURSOR_ON           0x0F  							// Cursor on       
#define CURSOR_OFF          0x0D  							// Cursor off      
#define BLINK_ON            0x0F  							// Cursor Blink    
#define BLINK_OFF           0x0E  							// Cursor No Blink 

/* Cursor or Display Shift */
#define SHIFT_CUR_LEFT      0x13  							// Cursor shifts to the left   
#define SHIFT_CUR_RIGHT     0x17  							// Cursor shifts to the right  
#define SHIFT_DISP_LEFT     0x1B  							// Display shifts to the left  
#define SHIFT_DISP_RIGHT    0x1F  							// Display shifts to the right 

/* pototype  section */
void init_uart1(void);										// Initial UART1 Function

void Initial_4bitLCD(void);									// Initial LCD Interface
void SetCGRamAddr(unsigned char);						
void SetDDRamAddr(unsigned char);							// Set Cursor Address
void WriteCmdLCD(unsigned char);							// Write Command
void WriteDataLCD(unsigned char);							// Write Data
void PutsLCD(unsigned char*);								// Print Message
void Delay_tW_LCD(void);									// Enable Pulse Delay	
void Busy_LCD(void);										// Wait LCD Busy
void Delay(unsigned long int);								// Delay Time Function

void Test_LCD(void);
void Test_UART(void);
void Test_GPIO(void);

int main(void)
{  
  char rx_data;												// Character Receive Buffer

  init_uart1();												// Initial UART = 9600,N,8,1
  
  Initial_4bitLCD();										// Initial LCD 4 Bit Interface
  SetDDRamAddr(0x00);										// Start Cursor Line-1   
  PutsLCD((unsigned char *)"ET-BASE dsPIC30F");
  SetDDRamAddr(0x40);										// Start Cursor Line-2   
  PutsLCD((unsigned char *)"(QC TEST System)");
  
  //UART1 Menu Test
  sprintf(uart_buf1,"\n\r\n\r\n\r\n\r");		
  putsUART1((unsigned int *)uart_buf1);                     
  while(BusyUART1());	
  sprintf(uart_buf1,"*******************************\n\r");		
  putsUART1((unsigned int *)uart_buf1);                     
  while(BusyUART1());										
  sprintf(uart_buf1,"ET-BASE dsPIC30F2010 QC Test\n\r"); 	
  putsUART1((unsigned int *)uart_buf1);                     
  while(BusyUART1());										
  sprintf(uart_buf1,"*******************************\n\r");		
  putsUART1((unsigned int *)uart_buf1);                    
  while(BusyUART1());										
  sprintf(uart_buf1,"1. Test LCD\n\r");						
  putsUART1((unsigned int *)uart_buf1);                     
  while(BusyUART1());										
  sprintf(uart_buf1,"2. Test UART\n\r");				
  putsUART1((unsigned int *)uart_buf1);                     
  while(BusyUART1());										
  sprintf(uart_buf1,"3. Test GPIO\n\r");					
  putsUART1((unsigned int *)uart_buf1);                     
  while(BusyUART1());										
  sprintf(uart_buf1,"*******************************\n\r");	
  putsUART1((unsigned int *)uart_buf1);                     
  while(BusyUART1());										
  sprintf(uart_buf1,"Select[1...3] For Test\n\r");  		
  putsUART1((unsigned int *)uart_buf1);                     
  while(BusyUART1());										
  sprintf(uart_buf1,"*******************************\n\r");		
  putsUART1((unsigned int *)uart_buf1);                     
  while(BusyUART1());										
  sprintf(uart_buf1,"Select...");  							
  putsUART1((unsigned int *)uart_buf1);                     
  while(BusyUART1());										

  while(1)													// Loop Continue
  {
    //UART1 Select Test
    if (DataRdyUART1())										// If RX Receive Complete
    {
      rx_data = ReadUART1();								// Get UART1 Data Receive
      switch (rx_data)
	  {
	    case '1' : putcUART1(rx_data);
                   while(BusyUART1());

                   sprintf(uart_buf1,"\n\r\n\rLCD Testing...\n\r");	
                   putsUART1((unsigned int *)uart_buf1);                     
                   while(BusyUART1());	
	 
		   		   Test_LCD();								// Test LCD 16x2
	               break;

	    case '2' : putcUART1(rx_data);
                   while(BusyUART1());

                   sprintf(uart_buf1,"\n\r\n\rUART Testing...\n\r\n\r");	
                   putsUART1((unsigned int *)uart_buf1);                     
                   while(BusyUART1());	

		  		   Test_UART();								// Test UART
	               break;

        case '3' : putcUART1(rx_data);
                   while(BusyUART1());

                   sprintf(uart_buf1,"\n\r\n\rGPIO Testing...\n\r");	
                   putsUART1((unsigned int *)uart_buf1);                     
                   while(BusyUART1());	
                   sprintf(uart_buf1,"Please Select Jumper RC13,RC14 = GPIO Mode...\n\r");	
                   putsUART1((unsigned int *)uart_buf1);                     
                   while(BusyUART1());

		  		   Test_GPIO();								// Test GPIO 
	               break;
      } 
    }

  } 
}	  

/*********************************/
/* Initial UART for dsPIC30F2010 */
/* 9600,N,8,1 / 117.9648MHz Fosc */
/*********************************/
void init_uart1()
{		  
  CloseUART1();												// Disable UART1 Before New Config

  // Config UART1 Interrupt Control
  ConfigIntUART1(UART_RX_INT_DIS &							// Disable RX Interrupt
    		     UART_RX_INT_PR2 &							// RX Interrupt Priority = 2
    		     UART_TX_INT_DIS &							// Disable TX Interrupt
    		     UART_TX_INT_PR3);							// TX Interrupt Priority = 3  
  // Open UART1 = Mode,Status,Baudrate              
  OpenUART1(UART_EN	&										// Enable UART(UART Mode)
            UART_IDLE_STOP &								// Disable UART in IDLE Mode 
 			UART_ALTRX_ALTTX & 								// Select U1TX=RC13,U1RX=RC14
            UART_DIS_WAKE &									// Disable Wake-Up
			UART_DIS_LOOPBACK &								// Disable Loop Back
			UART_DIS_ABAUD &								// Disable Auto Baudrate
  			UART_NO_PAR_8BIT &								// UART = 8 Bit, No Parity
 			UART_1STOPBIT,									// UART = 1 Stop Bit

	  		// Config UART1 Status
  			UART_INT_TX & 									// Select Interrupt After TX Complete
	 		UART_TX_PIN_NORMAL &							// Normal U1TX Mode
 			UART_TX_ENABLE &								// Enable U1TX
 	 		UART_INT_RX_BUF_FUL &							// Flasg Set After RX Complete 
  			UART_ADR_DETECT_DIS &              				// Disable Check Address 
			UART_RX_OVERRUN_CLEAR,							// Clear Overrun Flag

  			// ET-BASE dsPIC30F2010 Hardware Board
  			// XTAL = 7.3728MHz
  			// Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  			// Fcy(UART) = Fosc / 4 
  			//           = 117.9648 / 4 = 29.4912 MHz
  			// U1BRG = [Fcy/(16xBaud)]-1
  			//       = [29.4912 MHz / (16x9600)] - 1
  			//       = 191 = BFH			
  			191);											// ET-BASE dsPIC30F2010 UART Baudrate = 9600 BPS
}

void Initial_4bitLCD(void)
{
  TRIS_DATA_PIN_4 = 0;										// Set Port Direction = Output
  TRIS_DATA_PIN_5 = 0;
  TRIS_DATA_PIN_6 = 0;
  TRIS_DATA_PIN_7 = 0;
  TRIS_RS = 0;
  TRIS_E = 0;

  RS_PIN = 0;												// Instruction Select
  E_PIN = 0;												// Disable Strobe
  Delay(50000);												// Power-ON Delay (minimum of 15ms) 
  
  WriteCmdLCD(0x33); 										// Initial 4 Bit Interface Format  
  WriteCmdLCD(0x32);									
  WriteCmdLCD(0x28);										// Function Set (DL=0 4-Bit,N=1 2 Line,F=0 5X7)							

  WriteCmdLCD(0x0C);										// Display on/off Control (Entry Display,Cursor off,Cursor not Blink)
  WriteCmdLCD(0x06);  										// Entry Mode Set (I/D=1 Increment,S=0 Cursor Shift)	  
  WriteCmdLCD(0x01);    									// Clear Display  (Clear Display,Set DD RAM Address=0) 
}

void SetCGRamAddr(unsigned char address)
{
  RS_PIN = 0;  												// Select Instruction  
  DATA_PIN_7 = 0;											// DB7 Must be 0 For Setting CGRam ADDR
  DATA_PIN_6 = 1;											// DB6 Must be 1 For Setting CGRam ADDR  
  DATA_PIN_5 = ((address & 0x20)>>5);
  DATA_PIN_4 = ((address & 0x10)>>4);    
  E_PIN = 1;												// Strobe High Nibble
  Delay_tW_LCD();											// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();

  RS_PIN = 0;  												// Select Instruction
  DATA_PIN_7 = ((address & 0x08)>>3);
  DATA_PIN_6 = ((address & 0x04)>>2);
  DATA_PIN_5 = ((address & 0x02)>>1);
  DATA_PIN_4 = ((address & 0x01)>>0);
  E_PIN = 1;												// Strobe Low Nibble
  Delay_tW_LCD();											// Enable Pulse Delay
  E_PIN = 0;   
  Busy_LCD();  
}

void SetDDRamAddr(unsigned char address)
{  
  RS_PIN = 0;   											// Select Instruction 
  DATA_PIN_7 = 1;											// DB7 Must be 1 For Setting DDRam ADDR 
  DATA_PIN_6 = ((address & 0x40)>>6); 
  DATA_PIN_5 = ((address & 0x20)>>5);
  DATA_PIN_4 = ((address & 0x10)>>4);
  E_PIN = 1;												// Strobe High Nibble
  Delay_tW_LCD();											// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();
 
  RS_PIN = 0;   											// Select Instruction 
  DATA_PIN_7 = ((address & 0x08)>>3);	
  DATA_PIN_6 = ((address & 0x04)>>2);
  DATA_PIN_5 = ((address & 0x02)>>1);
  DATA_PIN_4 = ((address & 0x01)>>0);
  E_PIN = 1;    											// Strobe Low Nibble
  Delay_tW_LCD();											// Enable Pulse Delay
  E_PIN =0;
  Busy_LCD();
} 

void WriteCmdLCD(unsigned char cmd)
{
  RS_PIN = 0;   											// Select Instruction 
  DATA_PIN_7 = ((cmd & 0x80)>>7);
  DATA_PIN_6 = ((cmd & 0x40)>>6);
  DATA_PIN_5 = ((cmd & 0x20)>>5);
  DATA_PIN_4 = ((cmd & 0x10)>>4);
  E_PIN = 1;												// Strobe High Nibble
  Delay_tW_LCD();											// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();
  
  RS_PIN = 0;   											// Select Instruction 
  DATA_PIN_7 = ((cmd & 0x08)>>3);
  DATA_PIN_6 = ((cmd & 0x04)>>2);
  DATA_PIN_5 = ((cmd & 0x02)>>1);
  DATA_PIN_4 = (cmd & 0x01);
  E_PIN = 1;												// Strobe Low Nibble
  Delay_tW_LCD();											// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();
  Delay(50000);												// 1.64mS Delay	
}  

void WriteDataLCD(unsigned char data)
{
  RS_PIN = 1;  												// Select Data                    
  DATA_PIN_7 = ((data & 0x80)>>7);
  DATA_PIN_6 = ((data & 0x40)>>6);
  DATA_PIN_5 = ((data & 0x20)>>5);
  DATA_PIN_4 = ((data & 0x10)>>4);
  E_PIN = 1;												// Strobe High Nibble
  Delay_tW_LCD();											// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();

  RS_PIN = 1;  												// Select Data   
  DATA_PIN_7 = ((data & 0x08)>>3);
  DATA_PIN_6 = ((data & 0x04)>>2);
  DATA_PIN_5 = ((data & 0x02)>>1);
  DATA_PIN_4 = (data & 0x01);
  E_PIN = 1;												// Strobe Low Nibble
  Delay_tW_LCD();											// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();
}  

void PutsLCD(unsigned char* buffer)
{
  while(*buffer != '\0')									// Loop Until End of Message
  {    
    WriteDataLCD(*buffer); 									// Write Character to LCD
    buffer++;
  } 
}

void Delay_tW_LCD(void)   									// Enable Pulse Delay
{
  int i;
  for(i=0;i<18;i++);       
}

void Busy_LCD(void)											// 40uS Delay
{
  unsigned int i;
  for(i=0;i<1800;i++);      
}

/***********************/
/* Delay Time Function */
/*    1-4294967296     */
/***********************/
void Delay(unsigned long int count1)
{
  while(count1 > 0) {count1--;}								// Loop Decrease Counter	
}

/***************************/
/* Test Character LCD 16X2 */
/***************************/
void Test_LCD(void)
{
  Initial_4bitLCD();										// Initial LCD 4 Bit Interface

  while(1)
  {
    SetDDRamAddr(0x00);										// Start Cursor Line-1   
    PutsLCD((unsigned char *)"ET-BASE dsPIC30F");
    SetDDRamAddr(0x40);										// Start Cursor Line-2   
    PutsLCD((unsigned char *)"(QC TEST System)");
    Delay(5000000);											// Delay Display  

    SetDDRamAddr(0x00);										// Start Cursor Line-1   
    PutsLCD((unsigned char *)"16 Bit / 29 MIPS");
    SetDDRamAddr(0x40);										// Start Cursor Line-2   
    PutsLCD((unsigned char *)"BY...ETT CO.,LTD");
    Delay(5000000);											// Delay Display      
  }
}

/***********************/
/* Test UART Echo Test */
/***********************/
void Test_UART(void)
{
  char rx_data;												// Character Receive Buffer

  init_uart1();												// Initial UART = 9600,N,8,1
  
  //UART1
  sprintf(uart_buf1,"ET-BASE dsPIC30F2010 UART1\n\r"); 		// Print Message String 
  putsUART1((unsigned int *)uart_buf1);                     // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete
  sprintf(uart_buf1,"UART1(RS232) Echo Test\n\n\r");		// Print Message String 
  putsUART1((unsigned int *)uart_buf1);                     // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete  
  sprintf(uart_buf1,"Press Any Key For Test\n\r"); 			// Print Message String 
  putsUART1((unsigned int *)uart_buf1);                     // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete  

  while(1)													// Loop Continue
  {
    //UART1
    if (DataRdyUART1())										// If RX Receive Complete
    {
      rx_data = ReadUART1();								// Get UART1 Data Receive
      if (rx_data == 0x0D)									// If Receive Enter
      {
        sprintf(uart_buf1,"UART1(RS232)\n\r");				// Print Message String 
        putsUART1((unsigned int *)uart_buf1);               // Print uart_buff to UART1
        while(BusyUART1());									// Wait putsUART1 Complete  
      } 
      putcUART1(rx_data);									// Echo to TX of UART1
      while(BusyUART1());									// Wait putcUART1 Complete  
    }
  } 
}

/**************************/
/* Test GPIO (Output) Pin */
/**************************/
void Test_GPIO(void)
{  
  TRISBbits.TRISB0 = 0;										// Config RB0 = Output
  TRISBbits.TRISB1 = 0;										// Config RB1 = Output
  TRISBbits.TRISB2 = 0;										// Config RB2 = Output
  TRISBbits.TRISB3 = 0;										// Config RB3 = Output
  TRISBbits.TRISB4 = 0;										// Config RB4 = Output
  TRISBbits.TRISB5 = 0;										// Config RB5 = Output
  
  TRISEbits.TRISE0 = 0;										// Config RE0 = Output
  TRISEbits.TRISE1 = 0;										// Config RE1 = Output
  TRISEbits.TRISE2 = 0;										// Config RE2 = Output
  TRISEbits.TRISE3 = 0;										// Config RE3 = Output
  TRISEbits.TRISE4 = 0;										// Config RE4 = Output
  TRISEbits.TRISE5 = 0;										// Config RE5 = Output
  TRISEbits.TRISE8 = 0;										// Config RE8 = Output
  
  TRISFbits.TRISF2 = 0;										// Config RF2 = Output
  TRISFbits.TRISF3 = 0;										// Config RF3 = Output
  
  TRISDbits.TRISD0 = 0;										// Config RD0 = Output
  TRISDbits.TRISD1 = 0;										// Config RD1 = Output

  //Disable U1TX=RC13,U1RX=RC14 to GPIO Mode
  CloseUART1();

  TRISCbits.TRISC13 = 0;									// Config RC13 = Output
  TRISCbits.TRISC14 = 0;									// Config RC14 = Output
 
  // Loop Moving LED on Rx[0..7] //
  while(1)													// Loop Continue
  {   
    LATBbits.LATB0 = 0; 	    
    LATEbits.LATE0 = 0; 	 
    LATCbits.LATC13 = 0; 
    Delay(1500000);											// Display Delay
    LATBbits.LATB0 = 1; 	     
    LATEbits.LATE0 = 1; 	 
    LATCbits.LATC13 = 1; 	
 
    LATBbits.LATB1 = 0; 	    
    LATEbits.LATE1 = 0; 	    	
    LATCbits.LATC14 = 0; 
    Delay(1500000);											// Display Delay
    LATBbits.LATB1 = 1; 	    
    LATEbits.LATE1 = 1; 	
    LATCbits.LATC14 = 1; 

    LATBbits.LATB2 = 0; 	    
    LATEbits.LATE2 = 0; 	 
    LATFbits.LATF2 = 0;         	
    LATDbits.LATD0 = 0; 	 
    Delay(1500000);											// Display Delay
    LATBbits.LATB2 = 1; 	    	 
    LATEbits.LATE2 = 1; 	 
    LATFbits.LATF2 = 1;
    LATDbits.LATD0 = 1; 

    LATBbits.LATB3 = 0; 	    	 
    LATEbits.LATE3 = 0; 	 
    LATFbits.LATF3 = 0;	          	
    LATDbits.LATD1 = 0; 
    Delay(1500000);											// Display Delay
    LATBbits.LATB3 = 1; 	    
    LATEbits.LATE3 = 1; 	 
    LATFbits.LATF3 = 1;
    LATDbits.LATD1 = 1; 

    LATBbits.LATB4 = 0; 	    	 
    LATEbits.LATE4 = 0; 	 
    Delay(1500000);											// Display Delay
    LATBbits.LATB4 = 1; 	    
    LATEbits.LATE4 = 1; 	 
    
    LATBbits.LATB5 = 0; 	    	 
    LATEbits.LATE5 = 0; 	  	
    Delay(1500000);											// Display Delay
    LATBbits.LATB5 = 1; 	    	 
    LATEbits.LATE5 = 1; 	 
       
    LATEbits.LATE8 = 0; 	        	
    Delay(1500000);											// Display Delay
    LATEbits.LATE8 = 1; 	 
  }   
}	  
